unit utuOQL;

interface

uses utuMessage, acuContainers, acuObject;

const
  cOQLQuery = 0;
  cDataSetOQLDataSet = 1;

type
  utOQL = Class;

  utOQLFile = class
  private
    fOQLs: utField;
    fOQLHash: acObjectHash;
    fFileName: string;
    procedure AddOQL(piOQL: utOQL);
    procedure Load;
    procedure ChangeOQLName(piNewName: string; piOQL: utOQL);
  public
    constructor create(piFileName: string = '');
    destructor Destroy; override;
    function OQLExists(piName: string): boolean;
    function NewOQL(piOQLName: string): utOQL;
    procedure DeleteOQL(piName: string);
    procedure Save(piFileName: string = '');
    function Count: integer;
    function GetOQLEnumerator: acEnumerator;
    property FileName: string read fFileName;
  end;

  utOQL = class(utField)
  private
    fOQLFile: utOQLFile;
    function GetDescription: string;
    procedure SetDescription(const Value: string);
    function GetQuery: string;
    procedure SetQuery(const Value: string);
    procedure SetOwner(const Value: utOQLFile);
    function GetID: string;
    procedure SetID(const Value: string);
    function GetOQLType: integer;
    procedure SetOQLType(const Value: integer);
  public
    constructor Create(piOQLName: string; piOwner: utOQLFile); reintroduce;
    procedure Delete;
    property Owner: utOQLFile read fOQLFile write SetOwner;
    property Description: string read GetDescription write SetDescription;
    property Query: string read GetQuery write SetQuery;
    property ID: string read GetID write SetID;
    property OQLType: integer read GetOQLType write SetOQLType;
  end;


implementation

uses SysUtils, Classes;

{ utOQLFile }

procedure utOQLFile.AddOQL(piOQL: utOQL);
begin
  if assigned(fOQLHash.ValueOf(piOQL.FindAttribute('name').AsString)) then
    raise Exception.Create('OQL already exists with this name.');

  Self.fOQLs.AddField(piOQL);
  fOQLHash.Add(piOQL.AttributeByName('name').AsString, piOQL);
end;

procedure utOQLFile.ChangeOQLName(piNewName: string; piOQL: utOQL);
begin
  if piNewName <> piOQL.AttributeByName('name').AsString then
  begin
    if self.OQLExists(piNewName) then
      raise Exception.Create('OQL name already exists.');
    fOQLHash.Remove(piOQL.AttributeByName('name').AsString);
    fOQLHash.Add(piNewName, piOQL);
    piOQL.AttributeByName('name').AsString := piNewName;
  end;
end;

function utOQLFile.Count: integer;
begin
  result := fOQLHash.Count;
end;

constructor utOQLFile.Create(piFileName: string = '');
begin
  fFileName := piFileName;
  if (fFileName <> '') and (not FileExists(fFileName)) then
    raise Exception.Create('File does not exist.')
  else
  begin
    fOQLs := utField.Create;
    fOQLHash := acObjectHash.Create(127, false, true);
    fOQLs.Name := 'DATA';
    if fFileName <> '' then Self.Load;
  end;
end;

procedure utOQLFile.DeleteOQL(piName: string);
var lField: utField;
begin
  lField := utField(fOQLHash.ValueOf(piName));
  if not assigned(lField) then
    raise Exception.Create('No OQL found with name "' + piName + '".');

  fOQLHash.Remove(piName);
  fOQLs.RemoveField(lField);
end;

function utOQLFile.NewOQL(piOQLName: string): utOQL;
begin
  result := utOQL.Create(piOQLName, self);
end;

function utOQLFile.OQLExists(piName: string): boolean;
begin
  result := Assigned(fOQLHash.ValueOf(piName));
end;

procedure utOQLFile.Save(piFileName: string);
var lEnum: acEnumerator;
    lOQL: utOQL;
    lSortedField, lOQLField: utField;
begin
  if (piFileName = '') and (self.fFileName = '') then
    raise Exception.Create('No file name.');
  if piFileName <> '' then self.fFileName := piFileName;

  //Salva Ordenado
  lSortedField := utField.Create;
  try
    lSortedField.Name := 'DATA';
    lEnum := fOQLHash.GetEnumerator;
    try
      while not lEnum.EOL do
      begin
        lOQL := utOQL(lEnum.Current);

        lOQLField := lSortedField.AddField('oql');
        lOQLField.AddAttribute('name').AsString := lOQL.ID;
        lOQLField.AddAttribute('oqltype').AsInteger := lOQL.OQLType;
        lOQLField.AddField('description').AsString := lOQL.Description;
        lOQLField.AddField('query').AsCDATA := lOQL.Query;

        lEnum.MoveNext;
      end;

      lSortedField.SaveToFile(self.fFileName, XMLEncoding.xeISO_8859_1, [XMLOptions.xoLegible]);
    finally
      lEnum.Free;
    end;
  finally
    lSortedField.Free;
  end;
end;

destructor utOQLFile.destroy;
begin
  fOQLs.Free;
  fOQLHash.Free;
  inherited;
end;

function utOQLFile.GetOQLEnumerator: acEnumerator;
begin
  result := fOQLs.GetFieldsEnumerator;
end;

procedure utOQLFile.Load;
var lEnum: acEnumerator;
    lFields, lFieldOQL: utField;
    lOQL: utOQL;
    lXML: TStringList;
begin
  lXML := TStringList.Create;
  try
    lXML.LoadFromFile(fFileName);
    lFields := utField.Create;
    try
      lFields.LoadFieldFromXMLString(lXML.Text);
      lEnum := lFields.GetFieldsEnumerator;
      try
        while not lEnum.EOL do
        begin
          lFieldOQL := utField(lEnum.Current);
          lOQL := self.NewOQL(lFieldOQL.AttributeByName('name').AsString);
          lOQL.Description := lFieldOQL.FieldByName('description').AsString;
          lOQL.Query := lFieldOQL.FieldByName('query').AsString;

          if lFieldOQL.HasAttribute('oqltype')
            then lOQL.OQLType := lFieldOQL.AttributeByName('oqltype').AsInteger
            else lOQL.OQLType := cOQLQuery;
            
          lEnum.MoveNext;
        end;
      finally
        lEnum.free;
      end;
    finally
      lFields.Free;
    end;
  finally
    lXML.free;
  end;
end;

{ utOQL }

constructor utOQL.Create(piOQLName: string; piOwner: utOQLFile);
begin
  inherited create;
  self.Name := 'oql';
  self.AddAttribute('name').AsString := piOQLName;
  self.AddAttribute('oqltype').AsInteger := 0;
  self.AddField('description').AsString := '';
  self.AddField('query').AsCDATA := '';
  self.Owner := piOwner;
end;

procedure utOQL.Delete;
begin
  Self.fOQLFile.DeleteOQL(Self.ID);
end;

function utOQL.GetDescription: string;
begin
  result := Self.FieldByName('description').AsString;
end;

function utOQL.GetID: string;
begin
  result := self.AttributeByName('name').AsString;
end;

function utOQL.GetOQLType: integer;
begin
  result := self.AttributeByName('oqltype').AsInteger;
end;

function utOQL.GetQuery: string;
begin
  result := self.FieldByName('query').AsString;
end;

procedure utOQL.SetDescription(const Value: string);
begin
  Self.FieldByName('description').AsString := Value;
end;

procedure utOQL.SetID(const Value: string);
begin
  fOQLFile.ChangeOQLName(Value, self);
end;

procedure utOQL.SetOQLType(const Value: integer);
begin
  self.AttributeByName('oqltype').AsInteger := Value;
end;

procedure utOQL.SetOwner(const Value: utOQLFile);
begin
  fOQLFile := Value;
  Value.AddOQL(self);
end;

procedure utOQL.SetQuery(const Value: string);
begin
  self.FieldByName('query').AsCDATA := Value;
end;

end.
