unit utuXSD;

interface

uses utuMessage, acuContainers, acuObject;

type
  utXSD = Class;

  utXSDFile = class
  private
    fXSDs: utField;
    fXSDHash: acObjectHash;
    fFileName: string;
    procedure AddXSD(piXSD: utXSD);
    procedure Load(piXSD: string);
    procedure ChangeXSDName(piNewName: string; piXSD: utXSD);
    function GetXML: string;
  public
    constructor create(piXSD: string = '');
    destructor Destroy; override;
    function XSDExists(piName: string): boolean;
    function XSDByName(piName: string): utXSD;
    function NewXSD(piXSDName: string): utXSD;
    procedure DeleteXSD(piName: string);
    procedure Save(piFileName: string = '');
    function Count: integer;
    function GetXSDEnumerator: acEnumerator;
    property FileName: string read fFileName;
    property XML: string read GetXML;    
  end;

  utXSD = class(utField)
  private
    fXSDFile: utXSDFile;
    function GetDescription: string;
    procedure SetDescription(const Value: string);
    function GetText: string;
    procedure SetText(const Value: string);
    procedure SetOwner(const Value: utXSDFile);
    function GetID: string;
    procedure SetID(const Value: string);
  public
    constructor Create(piXSDName: string; piOwner: utXSDFile); reintroduce;
    procedure Delete;
    property Owner: utXSDFile read fXSDFile write SetOwner;
    property Description: string read GetDescription write SetDescription;
    property Text: string read GetText write SetText;
    property ID: string read GetID write SetID;
  end;


implementation

uses SysUtils, Classes;

{ utXSDFile }

procedure utXSDFile.AddXSD(piXSD: utXSD);
begin
  if assigned(fXSDHash.ValueOf(piXSD.FindAttribute('name').AsString)) then
    raise Exception.Create('XSD already exists with this name.');

  Self.fXSDs.AddField(piXSD);
  fXSDHash.Add(piXSD.AttributeByName('name').AsString, piXSD);
end;

procedure utXSDFile.ChangeXSDName(piNewName: string; piXSD: utXSD);
begin
  if piNewName <> piXSD.AttributeByName('name').AsString then
  begin
    if self.XSDExists(piNewName) then
      raise Exception.Create('XSD name already exists.');
    fXSDHash.Remove(piXSD.AttributeByName('name').AsString);
    fXSDHash.Add(piNewName, piXSD);
    piXSD.AttributeByName('name').AsString := piNewName;
  end;
end;

function utXSDFile.Count: integer;
begin
  result := fXSDHash.Count;
end;

constructor utXSDFile.Create(piXSD: string = '');
begin
  fXSDs := utField.Create;
  fXSDHash := acObjectHash.Create(127, false, true);
  fXSDs.Name := 'DATA';
  if piXSD <> '' then Self.Load(piXSD);
end;

procedure utXSDFile.DeleteXSD(piName: string);
var lField: utField;
begin
  lField := utField(fXSDHash.ValueOf(piName));
  if not assigned(lField) then
    raise Exception.Create('No XSD found with name "' + piName + '".');

  fXSDHash.Remove(piName);
  fXSDs.RemoveField(lField);
end;

function utXSDFile.NewXSD(piXSDName: string): utXSD;
begin
  result := utXSD.Create(piXSDName, self);
end;

function utXSDFile.XSDByName(piName: string): utXSD;
begin
  result := nil;
  if self.XSDExists(piName) then
    result := utXSD(fXSDHash.ValueOf(piName));
end;

function utXSDFile.XSDExists(piName: string): boolean;
begin
  result := Assigned(fXSDHash.ValueOf(piName));
end;

procedure utXSDFile.Save(piFileName: string);
var lEnum: acEnumerator;
    lXSD: utXSD;
    lSortedField, lXSDField: utField;
begin
  if (piFileName = '') and (self.fFileName = '') then
    raise Exception.Create('No file name.');
  if piFileName <> '' then self.fFileName := piFileName;

  //Salva Ordenado
  lSortedField := utField.Create;
  try
    lSortedField.Name := 'DATA';
    lEnum := fXSDHash.GetEnumerator;
    try
      while not lEnum.EOL do
      begin
        lXSD := utXSD(lEnum.Current);

        lXSDField := lSortedField.AddField('XSD');
        lXSDField.AddAttribute('name').AsString := lXSD.ID;
        lXSDField.AddField('description').AsString := lXSD.Description;
        lXSDField.AddField('text').AsCDATA := lXSD.Text;

        lEnum.MoveNext;
      end;

      lSortedField.SaveToFile(self.fFileName, XMLEncoding.xeISO_8859_1, [XMLOptions.xoLegible]);
    finally
      lEnum.Free;
    end;
  finally
    lSortedField.Free;
  end;
end;

destructor utXSDFile.destroy;
begin
  fXSDs.Free;
  fXSDHash.Free;
  inherited;
end;

function utXSDFile.GetXML: string;
var lEnum: acEnumerator;
    lXSD: utXSD;
    lSortedField, lXSDField: utField;
begin
  //Salva Ordenado
  lSortedField := utField.Create;
  try
    lSortedField.Name := 'DATA';
    lEnum := fXSDHash.GetEnumerator;
    try
      while not lEnum.EOL do
      begin
        lXSD := utXSD(lEnum.Current);

        lXSDField := lSortedField.AddField('XSD');
        lXSDField.AddAttribute('name').AsString := lXSD.ID;
        lXSDField.AddField('description').AsString := lXSD.Description;
        lXSDField.AddField('text').AsCDATA := lXSD.text;

        lEnum.MoveNext;
      end;

      result := lSortedField.GetXMLString([XMLOptions.xoLegible]);
    finally
      lEnum.Free;
    end;
  finally
    lSortedField.Free;
  end;
end;

function utXSDFile.GetXSDEnumerator: acEnumerator;
begin
  result := fXSDs.GetFieldsEnumerator;
end;

procedure utXSDFile.Load(piXSD: string);
var lEnum: acEnumerator;
    lFields, lFieldXSD: utField;
    lXSD: utXSD;
begin
  lFields := utField.Create;
  try
    lFields.LoadFieldFromXMLString(piXSD);
    lEnum := lFields.GetFieldsEnumerator;
    try
      while not lEnum.EOL do
      begin
        lFieldXSD := utField(lEnum.Current);
        lXSD := self.NewXSD(lFieldXSD.AttributeByName('name').AsString);
        lXSD.Description := lFieldXSD.FieldByName('description').AsString;
        lXSD.text := lFieldXSD.FieldByName('text').AsString;
        lEnum.MoveNext;
      end;
    finally
      lEnum.free;
    end;
  finally
    lFields.Free;
  end;
end;

{ utXSD }

constructor utXSD.Create(piXSDName: string; piOwner: utXSDFile);
begin
  inherited create;
  self.Name := 'XSD';
  self.AddAttribute('name').AsString := piXSDName;
  self.AddField('description').AsString := '';
  self.AddField('text').AsCDATA := '';
  self.Owner := piOwner;
end;

procedure utXSD.Delete;
begin
  Self.fXSDFile.DeleteXSD(Self.ID);
end;

function utXSD.GetDescription: string;
begin
  result := Self.FieldByName('description').AsString;
end;

function utXSD.GetID: string;
begin
  result := self.AttributeByName('name').AsString;
end;

function utXSD.GetText: string;
begin
  result := self.FieldByName('text').AsString;
end;

procedure utXSD.SetDescription(const Value: string);
begin
  Self.FieldByName('description').AsString := Value;
end;

procedure utXSD.SetID(const Value: string);
begin
  fXSDFile.ChangeXSDName(Value, self);
end;

procedure utXSD.SetOwner(const Value: utXSDFile);
begin
  fXSDFile := Value;
  Value.AddXSD(self);
end;

procedure utXSD.SetText(const Value: string);
begin
  self.FieldByName('text').AsCDATA := Value;
end;

end.
