unit ituOQLExplorerV2;

{$IFDEF FPC}
  {$MODE Delphi}
{$ENDIF}

interface

uses
{$IFnDEF FPC}
  Windows,
{$ELSE}
  LCLIntf, LCLType, LMessages,
{$ENDIF}
  Messages, SysUtils, Variants, Classes, Graphics, Controls, Forms,
  Dialogs, ExtCtrls, StdCtrls, utuOQL, Menus, ComCtrls, Buttons, ActnList, ituExplorerV2, ImgList,
  ituDataBaseLogin, ToolWin, SynHighlighterPas, SynEdit;

type

  { TOQLExplorerV2 }

  TOQLExplorerV2 = class(TForm)
    abCXOQL: TAction;
    abExportPAS: TAction;
    acttbOQL: TAction;
    aExportPas: TAction;
    aSaveFile: TAction;
    aLoadFile: TAction;
    aNewFile: TAction;
    aDeleteOQL: TAction;
    aNewOQL: TAction;
    ActionList1: TActionList;
    ilCXOQL: TImageList;
    pcCXOQL: TPageControl;
    SelectUnitDirectory: TSelectDirectoryDialog;
    memUnit: TSynEdit;
    SynFreePascalSyn1: TSynFreePascalSyn;
    ToolBar1: TToolBar;
    ToolBar2: TToolBar;
    ToolButton1: TToolButton;
    ToolButton2: TToolButton;
    ToolButton3: TToolButton;
    ToolButton4: TToolButton;
    ToolButton5: TToolButton;
    ToolButton6: TToolButton;
    tsCXOQL: TTabSheet;
    tsPASunit: TTabSheet;
    ilOQLState: TImageList;
    pnlCXOQL: TPanel;
    pnlOQLList: TPanel;
    lvOQLList: TListView;
    memError: TMemo;
    splVertical: TSplitter;
    pnlRight: TPanel;
    pnlOQLExplorer: TPanel;
    pnlTop: TPanel;
    labelName: TLabel;
    labelDescription: TLabel;
    edtName: TEdit;
    edtDescription: TEdit;
    pnlPASUnit: TPanel;
    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    pnlList: TPanel;
    procedure FormClose(Sender: TObject; var Action: TCloseAction);
    procedure aNewOQLExecute(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure aDeleteOQLExecute(Sender: TObject);
    procedure aSaveFileExecute(Sender: TObject);
    procedure aLoadFileExecute(Sender: TObject);
    procedure aExportPASExecute(Sender: TObject);
    procedure FormShow(Sender: TObject);
    procedure FormCloseQuery(Sender: TObject; var CanClose: Boolean);
    procedure FormResize(Sender: TObject);
    procedure lvOQLListChange(Sender: TObject; Item: TListItem;Change: TItemChange);
    procedure aNewFileExecute(Sender: TObject);
    procedure splVerticalCanResize(Sender: TObject; var NewSize: Integer; var Accept: Boolean);

  protected
    procedure CreateParams(var Params : TCreateParams); override;

  private
    { Private declarations }
    fFlagFormClosing : Boolean; // Impede que as OQLs sejam inutilmente verificados ao fechar o form
    fExplorer: TExplorerV2;
    fOQLList: utOQLFile;
    fLastItem: TListItem;
    function GerarUnitOQL(piXMLFileName: String): string;
    procedure InterfaceToSelectedOQL;
    function GetQueryType ( piExplorer : TExplorerV2 ) : Integer;
    function ValidaQuery(piQuery: string; piOQLType: integer): string;
    procedure SetQueryType (piType: integer);
    procedure SelectedOQLToInterface;
    procedure UpdateOQLList;
  public
    { Public declarations }
    constructor Create(piOwner: TComponent); override;

  const
    __OQLQuery = 0;
    __XMLQuery = 1;
  end;

var
  OQLExplorerV2: TOQLExplorerV2;

implementation

uses acuObject, acuOQL, acuOQLtoSQLTranslator, IntegracaoDelphiSUML, utuMessage;

{$R *.dfm}

procedure TOQLExplorerV2.aDeleteOQLExecute(Sender: TObject);
var li: integer;
    lOQL: utOQL;
begin
  if assigned(lvOQLList.Selected) then
  begin
    lOQL := utOQL(lvOQLList.Selected.Data);
    li := lvOQLList.Selected.Index;

    lvOQLList.Selected.Delete;
    lOQL.Delete;
    fLastItem := Nil;
    if lvOQLList.Items.Count > 0 then
      if li - 1 >= 0 then
        lvOQLList.Selected := lvOQLList.Items[li-1]
      else
        lvOQLList.Selected := lvOQLList.Items[0];
      SelectedOQLToInterface;
  end;
end;

procedure TOQLExplorerV2.aExportPASExecute(Sender: TObject);
var lFileName: string;
begin
  if fOQLList.FileName = '' then
    ShowMessage('You need to save before exporting.')
  else
  begin
    InterfaceToSelectedOQL;
    fOQLList.Save;
    Screen.Cursor := crHourGlass;
    try
      memUnit.Lines.Text := GerarUnitOQL(fOQLList.FileName);
    finally
      Screen.Cursor := crDefault;
    end;
    pcCXOQL.ActivePage := tsPASUnit;
  end;
end;

function TOQLExplorerV2.GerarUnitOQL(piXMLFileName: String): string;

  procedure AddText(piStringStream: TStringStream; piIndent: integer = 0; piText: string = '');
  begin
    piText := StringOfChar(' ', piIndent) + piText;
    piStringStream.WriteString(concat(piText,#13#10));
  end;

var
  lUnit: TStringStream;
  lUnitFile: TFileStream;
  lFieldOQLFile, lFieldOQL: utField;
  lAux: TStringList;
  lOQLName, lOQLQuery: string;
  lEnum, lEnumParams: acEnumerator;
  lOQLParam: acOQLParam;
  lOQLtoSQLTranslator: acOQLtoSQLTranslator;
  lQueryParams: acQueryParams;
  lFileName: string;
  lOQLType: integer;
  lSQL: string;
begin
  if not FileExists(piXMLFileName) then
    raise Exception.Create('File "' + piXMLFileName + '" does not exist');

  lUnit := TStringStream.Create('');
  try
    lFieldOQLFile := utField.Create;
    try
      lAux := TStringList.Create;
      try
        lAux.LoadFromFile(piXMLFileName);
        lFieldOQLFile.LoadFieldFromXMLString(lAux.Text);
      finally
        lAux.free;
      end;

      lFileName := ExtractFileName(piXMLFileName);
      lFileName := LeftStr(lFileName, pos('.', lFileName) - 1);

      AddText(lUnit, 0, 'unit utu' + lFileName + ';');
      AddText(lUnit);
      AddText(lUnit, 0, 'interface');
      AddText(lUnit);
      AddText(lUnit, 0, 'uses acuOQL, acuFramework;');
      AddText(lUnit);
      AddText(lUnit, 0, 'type');
      AddText(lUnit);

      lEnum := lFieldOQLFile.GetFieldsEnumerator;
      try
        //OQL Interfaces
        while not lEnum.EOL do
        begin
          lFieldOQL := utField(lEnum.Current);
          if not lFieldOQL.HasAttribute('name') then raise Exception.Create('Attribute "name" not found for OQL. Not a valid XML OQL file.');
          lOQLName := lFieldOQL.AttributeByName('name').AsString;
          if not lFieldOQL.HasField('query') then raise Exception.Create('Field "query" not found for OQL. Not a valid XML OQL file.');
          lOQLQuery := lFieldOQL.FieldByName('query').AsString;

          //Default cOQLQuery
          lOQLType := cOQLQuery;
          if lFieldOQL.HasAttribute('oqltype') then
            lOQLType := lFieldOQL.AttributeByName('oqltype').AsInteger;

          case lOQLType of
            cDataSetOQLDataSet: AddText(lUnit, 2, 'I' + lOQLName + ' = interface(IDataSetOQLQuery)');
                           else AddText(lUnit, 2, 'I' + lOQLName + ' = interface(IOQLQuery)');
          end;

          lQueryParams := acQueryParams.Create;
          try
            lOQLtoSQLTranslator := acOQLtoSQLTranslator.Create(fExplorer.Util.MetaModelPersistenceMap);
            try
              case lOQLType of
                cDataSetOQLDataSet:
                  begin
                    lSQL := lOQLtoSQLTranslator.TranslateOQLtoSQL(lOQLQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkDataSet, lQueryParams);
                  end;
                else
                  begin
                    lSQL := lOQLtoSQLTranslator.TranslateOQLtoSQL(lOQLQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkObject, lQueryParams);
                  end;
              end;
            finally
              lOQLtoSQLTranslator.Free;
            end;

            lEnumParams := lQueryParams.GetEnumerator;
            try
              while not lEnumParams.EOL do
              begin
                lOQLParam := acOQLParam(lEnumParams.Current);
                AddText(lUnit, 4, 'function Param_' + lOQLParam.Name + ': ' + lOQLParam.ClassName + ';');
                lEnumParams.MoveNext;
              end;
            finally
              lEnumParams.free;
            end;
            AddText(lUnit, 2, 'end;');
            AddText(lUnit);
          finally
            lQueryParams.Free;
          end;
          lEnum.MoveNext;
        end;

        //Factory Class Interface
        AddText(lUnit, 2, 'ut' + lFileName + ' = class');
        lEnum.MoveFirst;
        while not lEnum.EOL do
        begin
          lFieldOQL := utField(lEnum.Current);
          lOQLName := lFieldOQL.AttributeByName('name').AsString;

          AddText(lUnit, 4, 'class function ' + lOQLName + '(piSessao: acPersistenceSession): ' + 'I' + lOQLName + ';');

          lEnum.MoveNext;
        end;
        AddText(lUnit, 2, 'end;');
        AddText(lUnit);
        AddText(lUnit, 0, 'implementation');
        AddText(lUnit);
        AddText(lUnit, 0, 'type');
        AddText(lUnit);

        //OQL Types
        lEnum.MoveFirst;
        while not lEnum.EOL do
        begin
          lFieldOQL := utField(lEnum.Current);
          lOQLName := lFieldOQL.AttributeByName('name').AsString;
          lOQLQuery := lFieldOQL.FieldByName('query').AsString;

          //Default cOQLQuery
          lOQLType := cOQLQuery;
          if lFieldOQL.HasAttribute('oqltype') then
            lOQLType := lFieldOQL.AttributeByName('oqltype').AsInteger;


          case lOQLType of
            cDataSetOQLDataSet: AddText(lUnit, 2, 'T' + lOQLName + ' = class(acDataSetOQLQuery, ' + 'I' + lOQLName + ')');
            else AddText(lUnit, 2, 'T' + lOQLName + ' = class(acOQLQuery, ' + 'I' + lOQLName + ')');
          end;

          lQueryParams := acQueryParams.Create;
          try
            lOQLtoSQLTranslator := acOQLtoSQLTranslator.Create(fExplorer.Util.MetaModelPersistenceMap);
            try
              case lOQLType of
                cDataSetOQLDataSet:
                  begin
                    lSQL := lOQLtoSQLTranslator.TranslateOQLtoSQL(lOQLQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkDataSet, lQueryParams);
                  end;
                else
                  begin
                    lSQL := lOQLtoSQLTranslator.TranslateOQLtoSQL(lOQLQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkObject, lQueryParams);
                  end;
              end;
            finally
              lOQLtoSQLTranslator.Free;
            end;

            lEnumParams := lQueryParams.GetEnumerator;
            try
              while not lEnumParams.EOL do
              begin
                lOQLParam := acOQLParam(lEnumParams.Current);
                AddText(lUnit, 4, 'function Param_' + lOQLParam.Name + ': ' + lOQLParam.ClassName + ';');
                lEnumParams.MoveNext;
              end;
            finally
              lEnumParams.free;
            end;
            AddText(lUnit, 2, 'end;');
            AddText(lUnit);
          finally
            lQueryParams.Free;
          end;
          lEnum.MoveNext;
        end;

        //OQL Types Implementation
        lEnum.MoveFirst;
        while not lEnum.EOL do
        begin
          lFieldOQL := utField(lEnum.Current);
          lOQLName := lFieldOQL.AttributeByName('name').AsString;
          lOQLQuery := lFieldOQL.FieldByName('query').AsString;

          //Default cOQLQuery
          lOQLType := cOQLQuery;
          if lFieldOQL.HasAttribute('oqltype') then
            lOQLType := lFieldOQL.AttributeByName('oqltype').AsInteger;

          AddText(lUnit, 0, '{ T' + lOQLName + ' }');
          AddText(lUnit);

          lQueryParams := acQueryParams.Create;
          try
            lOQLtoSQLTranslator := acOQLtoSQLTranslator.Create(fExplorer.Util.MetaModelPersistenceMap);
            try
              case lOQLType of
                cDataSetOQLDataSet:
                  begin
                    lSQL := lOQLtoSQLTranslator.TranslateOQLtoSQL(lOQLQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkDataSet, lQueryParams);
                  end;
                else
                  begin
                    lSQL := lOQLtoSQLTranslator.TranslateOQLtoSQL(lOQLQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkObject, lQueryParams);
                  end;
              end;
            finally
              lOQLtoSQLTranslator.Free;
            end;

            lEnumParams := lQueryParams.GetEnumerator;
            try
              while not lEnumParams.EOL do
              begin
                lOQLParam := acOQLParam(lEnumParams.Current);
                AddText(lUnit, 0, 'function T' + lOQLName + '.Param_' + lOQLParam.Name + ': ' + lOQLParam.ClassName + ';');
                AddText(lUnit, 0,'begin');
                AddText(lUnit, 2,'result := ' + lOQLParam.ClassName + '(Self.ParamByName(' + QuotedStr(lOQLParam.Name) + '));');
                AddText(lUnit, 0,'end;');
                AddText(lUnit);
                lEnumParams.MoveNext;
              end;
            finally
              lEnumParams.free;
            end;
          finally
            lQueryParams.Free;
          end;

          lEnum.MoveNext;
        end;


        //Factory Class Implementation
        AddText(lUnit, 0, '{ utOQLs }');
        AddText(lUnit);
        lEnum.MoveFirst;
        while not lEnum.EOL do
        begin
          lFieldOQL := utField(lEnum.Current);
          lOQLName := lFieldOQL.AttributeByName('name').AsString;
          lOQLQuery := lFieldOQL.FieldByName('query').AsString;

          AddText(lUnit, 0, 'class function ut' + lFileName  + '.' + lOQLName + '(piSessao: acPersistenceSession): I' + lOQLName + ';');
          AddText(lUnit, 0, 'begin');
          AddText(lUnit, 2, 'result := T' + lOQLName + '.Create(pisessao,');

          lOQLQuery := StringReplace(lOQLQuery, '''', '''''', [rfReplaceAll]);
          lOQLQuery := StringReplace(lOQLQuery, sLineBreak, ''' + sLineBreak +' + sLineBreak + '''', [rfReplaceAll]);

          AddText(lUnit, 0, '''' + lOQLQuery + ''');');
          AddText(lUnit, 0, 'end;');
          AddText(lUnit);

          lEnum.MoveNext;
        end;

      finally
        lEnum.free;
      end;

      AddText(lUnit, 0, 'end.');

      result := lUnit.DataString;

      lFileName := 'utu' + lFileName + '.pas';
      SelectUnitDirectory.FileName := lFileName;

      if SelectUnitDirectory.Execute then
      begin
        try
          lUnitFile := TFileStream.Create(SelectUnitDirectory.FileName + DirectorySeparator + lFileName, fmCreate);
          lUnit.Position := 0;
          lUnitFile.CopyFrom(lUnit, lUnit.Size);
        finally
          lUnitFile.Free;
        end;
      end

    finally
      lFieldOQLFile.free;
    end;
  finally
    lUnit.free;
  end;
end;

procedure TOQLExplorerV2.aLoadFileExecute(Sender: TObject);
var
  lM: integer;
  lCancel: Boolean;
begin
  lCancel := False;

  if (lvOQLList.Items.Count > 1) or (edtName.Caption <> '_NEW_OQL_1') then
  begin
    lM := application.MessageBox('Save before open another file?', 'Attention', MB_ICONQUESTION + MB_YESNO);
    case lM of
      IDYES: aSaveFileExecute(Sender);
      IDCANCEL: lCancel := True;
    end;
  end;

  if not (lCancel) then
  begin
    try
      try
        OpenDialog1.InitialDir := ExtractFilePath(gStarUMLApp.ProjectManager.FileName);
        if OpenDialog1.Execute then
        begin
          lvOQLList.ItemIndex := -1;
          if assigned(fOQLList) then freeandnil(fOQLList);
          fOQLList := utOQLFile.create(OpenDialog1.FileName);
        end;
      Except
        on e:exception do
        begin
          if assigned(fOQLList) then freeandnil(fOQLList);
          fOQLList := utOQLFile.create;
          raise;
        end;
      end;
    finally
      Caption := 'Curió OQL Studio - ' + fOQLList.FileName;
      UpdateOQLList;
    end;
  end;
end;

procedure TOQLExplorerV2.aNewFileExecute(Sender: TObject);
var
  lM: integer;
  lCancel: Boolean;
begin

  lCancel := False;

  lM := application.MessageBox('Save before create new file?', 'Attention', MB_ICONQUESTION + MB_YESNO);
  case lM of
    IDYES: aSaveFileExecute(Sender);
    IDCANCEL: lCancel := True;
  end;

  if not lCancel then
  begin
    try
      lvOQLList.ItemIndex := -1;
      if assigned(fOQLList) then freeandnil(fOQLList);
      fOQLList := utOQLFile.Create;
      UpdateOQLList;
      aNewOQLExecute(Sender);
    finally
      Caption := 'New...';
    end;
  end;
end;

procedure TOQLExplorerV2.aNewOQLExecute(Sender: TObject);
var lListItem: TListItem;
    lOQL: utOQL;
    lCont: integer;
begin
  InterfaceToSelectedOQL;
  lCont := 1;
  while fOQLList.OQLExists('_NEW_OQL_' + inttostr(lCont)) do
    Inc(lCont);

  lOQL := fOQLList.NewOQL('_NEW_OQL_' + inttostr(lCont));
  lListItem := lvOQLList.Items.Add;
  lListItem.Caption := lOQL.ID;
  lListItem.Data := lOQL;
  lListItem.ImageIndex := 2;
  lListItem.SubItems.Add('No OQL query');
  lvOQLList.Selected := lListItem;
  SelectedOQLToInterface;
  edtName.SetFocus;
  edtName.SelectAll;
end;

procedure TOQLExplorerV2.aSaveFileExecute(Sender: TObject);
var lFileName: string;
begin
  InterfaceToSelectedOQL;
  lFileName := fOQLList.FileName;
  try
    if lFileName = '' then
      if SaveDialog1.Execute then
      begin
        lFileName := SaveDialog1.FileName;
        if pos('.cxoql', lFileName) = 0 then
          lFileName := lFileName + '.cxoql';
      end
      else raise Exception.Create('File not saved.');

    lvOQLList.SortType := stText;
    lvOQLList.SortType := stNone;
    fOQLList.Save(lFileName);
    Caption := 'Curió OQL Studio - ' + lFileName;
    Application.MessageBox('File saved.', '', MB_ICONINFORMATION);
  except
    on E: Exception do
    begin
      Application.MessageBox(PAnsiChar(E.Message), '', MB_ICONEXCLAMATION);
    end;
  end;
end;

constructor TOQLExplorerV2.Create(piOwner: TComponent);
begin
  inherited Create(piOwner);
  fFlagFormClosing := False;
  fExplorer := TExplorerV2.Create(nil);
  fExplorer.Parent := pnlOQLExplorer;
  fExplorer.BorderIcons := [];
  fExplorer.BorderStyle := bsNone;
  fExplorer.Align := alClient;
  fExplorer.WindowState := wsMaximized;
  fExplorer.Position := poDefaultPosOnly;
  fExplorer.Show;
end;

procedure TOQLExplorerV2.CreateParams(var Params: TCreateParams);
begin
  inherited CreateParams(Params);
  Params.ExStyle   := Params.ExStyle or WS_EX_APPWINDOW;
end;

procedure TOQLExplorerV2.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  fFlagFormClosing := True;
  fExplorer.Close;
  Action := caFree;
end;

procedure TOQLExplorerV2.FormCloseQuery(Sender: TObject; var CanClose: Boolean);
var lM: integer;
begin
  if (lvOQLList.Items.Count > 1) or (edtName.Caption <> '_NEW_OQL_1') then
  begin
    lM := application.MessageBox('Save before close?', 'Attention', MB_ICONQUESTION + MB_YESNO);
    case lM of
      IDYES: aSaveFileExecute(sender);
      IDCANCEL: CanClose := false;
    end;
  end;
end;

procedure TOQLExplorerV2.FormCreate(Sender: TObject);
var lFileName: string;
begin
  lFileName := '';
  fOQLList := utOQLFile.create(lFileName);
  UpdateOQLList;
  Caption := 'Curió OQL Studio';
end;

procedure TOQLExplorerV2.FormResize(Sender: TObject);
begin
  fExplorer.Top := 0;
  fExplorer.Left := 0;
end;

procedure TOQLExplorerV2.FormShow(Sender: TObject);
begin
  pcCXOQL.ActivePage := tsCXOQL;
  if fOQLList.Count = 0 then
    aNewOQL.Execute;
end;

function TOQLExplorerV2.GetQueryType(piExplorer: TExplorerV2): Integer;
begin
  if piExplorer.OQLQuery.Checked = True then
    Result := __OQLQuery
  else
    Result := __XMLQuery;
end;

procedure TOQLExplorerV2.InterfaceToSelectedOQL;
var lOQL: utOQL;
    lErro: string;
begin
  if Assigned(fLastItem) and Assigned(fExplorer.SynEditOQL) then
  begin
    lOQL := utOQL(fLastItem.Data);
    lOQL.ID := edtName.Text;
    lOQL.Description := edtDescription.Text;
    lOQL.Query := fExplorer.SynEditOQL.Text;
    lErro := '';
    if not fFlagFormClosing then
    begin
      lOQL.OQLType := GetQueryType(fExplorer);
      lErro := ValidaQuery(lOQL.Query, lOQL.OQLType);
    end;
    if lErro = '' then
    begin
      if lOQL.OQLType = 1
        then fLastItem.ImageIndex := 1
        else fLastItem.ImageIndex := 0;
    end
    else
      fLastItem.ImageIndex := 2;
    fLastItem.Caption := edtName.Text;
    fLastItem.SubItems[0] := lErro;
  end;
end;

procedure TOQLExplorerV2.lvOQLListChange(Sender: TObject; Item: TListItem; Change: TItemChange);
begin
  if assigned(fLastItem) and (fLastItem <> Item) then InterfaceToSelectedOQL;
  SelectedOQLToInterface;
end;

procedure TOQLExplorerV2.SelectedOQLToInterface;
var lOQL: utOQL;
begin
  edtName.Text := '';
  edtDescription.Text := '';
  fExplorer.SynEditOQL.Text := '';
  if Assigned(lvOQLList.Selected) and Assigned( fExplorer.SynEditOQL) then
  begin
    fLastItem := lvOQLList.Selected;
    lOQL := utOQL(lvOQLList.Selected.Data);
    edtName.Text := lOQL.ID;
    edtDescription.Text := lOQL.Description;
    fExplorer.SynEditOQL.Text := lOQL.Query;
    memError.Text := lvOQLList.Selected.SubItems[0];
    SetQueryType(lOQL.OQLType);
  end
  else fLastItem := Nil;
  lvOQLList.Column[0].Caption := 'OQL''s (' + IntToStr(fOQLList.Count) + ')';
end;

procedure TOQLExplorerV2.SetQueryType(piType: integer);
begin
  if piType = __OQLQuery then
    fExplorer.OQLQuery.Execute
  else
    fExplorer.XMLQuery.Execute;
end;

procedure TOQLExplorerV2.splVerticalCanResize(Sender: TObject; var NewSize: Integer; var Accept: Boolean);
begin
  Accept := (NewSize > 190);  
end;

procedure TOQLExplorerV2.UpdateOQLList;
var lListItem: TListItem;
    lEnum: acEnumerator;
    lOQL: utOQL;
    lErro: string;
begin
  lvOQLList.Items.BeginUpdate;
  try
    lvOQLList.Clear;
    lEnum := fOQLList.GetOQLEnumerator;
    try
      while not lEnum.EOL do
      begin
        lOQL := utOQL(lEnum.Current);
        lListItem := lvOQLList.Items.Add;
        lListItem.Caption := lOQL.ID;
        lListItem.Data := lOQL;
        lErro := ValidaQuery(lOQL.Query, lOQL.OQLType);
        lListItem.SubItems.Add(lErro);
        if lErro = '' then
        begin
          if lOQL.OQLType = 1
            then lListItem.ImageIndex := 1
            else lListItem.ImageIndex := 0;
        end
        else
          lListItem.ImageIndex := 2;
        lEnum.MoveNext;
      end;
    finally
      lEnum.free;
    end;
  finally
    lvOQLList.Items.EndUpdate;
  end;
  if lvOQLList.Items.Count > 0 then
  begin
    lvOQLList.ItemIndex := 0;
    SelectedOQLToInterface;
  end;
end;

function TOQLExplorerV2.ValidaQuery(piQuery: string; piOQLType: integer): string;
var lOQLtoSQLTranslator: acOQLtoSQLTranslator;
begin  
  result := '';
  if not fFlagFormClosing then
  begin
    try
      if Assigned(fExplorer) then
      begin
        lOQLtoSQLTranslator := acOQLtoSQLTranslator.Create(fExplorer.Util.MetaModelPersistenceMap);
        try
          case piOQLType of
            cDataSetOQLDataSet:
              begin
                lOQLtoSQLTranslator.TranslateOQLtoSQL(piQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkDataSet);
              end;
            else
              begin
                lOQLtoSQLTranslator.TranslateOQLtoSQL(piQuery, fExplorer.SQLDialect, fExplorer.Schema, acQueryResultKind.qrkObject);
              end;
          end;
        finally
          lOQLtoSQLTranslator.Free;
        end;
      end;
    except
      on e:exception do
      begin
        Result := e.Message;
      end;
    end;
  end;
end;

end.
