import React, { useState } from 'react'

import { people } from '@agiliza/__mocks__/proposalData/involvedPeople'
import { InvolvedPerson, InvolvedPersonType } from '@agiliza/api/domain'
import { INVOLVED_PERSON_IDS, INVOLVED_PERSON_NAMES } from '@agiliza/constants/involvedPeople'
import { maskCPFCNPJ } from '@agiliza/utils/masks'
import { Avatar, List, ListItem, ListItemAvatar, ListItemText, Typography } from '@material-ui/core'
import { withStyles, WithStyles } from '@material-ui/core/styles'
import { Add as AddIcon, Person as PersonIcon } from '@material-ui/icons'

import AddUpdateDialog from './AddUpdateDialog'
import SelectTypeDialog from './SelectTypeDialog'
import styles from './styles'

type ExtendedProps = WithStyles<typeof styles>

interface Props extends ExtendedProps {}

const Identification = (props: Props) => {
  const { classes } = props

  const [open, setOpen] = useState({ type: false, form: false })
  const [type, setType] = useState<InvolvedPersonType | ''>('')
  const [selectedPerson, setSelectedPerson] = useState<InvolvedPerson | undefined>()

  const handleSelectType = (type: InvolvedPersonType) => () => {
    setType(type)
    setSelectedPerson(undefined)
    setOpen({ type: false, form: true })
  }

  const handleClickAdd = () => {
    setOpen({ ...open, type: true })
  }

  const handleClickListItem = (person: InvolvedPerson) => () => {
    setSelectedPerson(person)
    setType(INVOLVED_PERSON_IDS[person.identification.type])
    setOpen({ ...open, form: true })
  }

  const handleSave = (person: InvolvedPerson) => () => {}

  return (
    <>
      <div className={classes.contentContainer}>
        <div className={classes.content}>
          <Typography className={classes.title}>Identicação</Typography>
          <List className={classes.list}>
            {people.map((person, i) => (
              <ListItem key={i} button onClick={handleClickListItem(person)}>
                <ListItemAvatar>
                  <Avatar>
                    <PersonIcon />
                  </Avatar>
                </ListItemAvatar>
                <ListItemText
                  primary={INVOLVED_PERSON_NAMES[person.identification.type]}
                  secondary={`${person.identification.name} - ${maskCPFCNPJ(person.identification.cpfcnpj)}`}
                />
              </ListItem>
            ))}
          </List>
        </div>
      </div>
      <Avatar
        className={classes.avatarBtnContainer}
        onClick={handleClickAdd}
        // rounded
        // Component={TouchableOpacity}
      >
        <AddIcon />
      </Avatar>
      <SelectTypeDialog open={open.type} onClose={() => setOpen({ ...open, type: false })} onSelectType={handleSelectType} />
      <AddUpdateDialog
        open={open.form}
        onClose={() => setOpen({ ...open, form: false })}
        person={selectedPerson}
        type={type as InvolvedPersonType}
        onSave={handleSave}
      />
    </>
  )
}

export default withStyles(styles)(Identification)
