import { extractCurrency, extractDecimal, extractNumbers } from './extractors'

// import { getUtcToZonedTime } from './methods'

export const formatPhone = (text: string) => {
  if (!text) return ''
  const filteredString = extractNumbers(text).slice(0, 11)
  const length = filteredString.length
  if (length < 3) return filteredString
  if (length < 8) return filteredString.replace(/(\d\d)/, '($1) ')
  return filteredString.replace(/(\d\d)(\d\d\d\d\d)/, '($1) $2-')
}

export const formatCPFCNPJ = (text: string) => {
  if (!text) return ''
  const filteredText = extractNumbers(text).slice(0, 14)
  const length = filteredText.length
  if (length < 4) return filteredText
  if (length < 7) return filteredText.replace(/(\d{3})/, '$1.')
  if (length < 10) return filteredText.replace(/(\d{3})(\d{3})/, '$1.$2.')
  if (length < 12) return filteredText.replace(/(\d{3})(\d{3})(\d{3})/, '$1.$2.$3-')
  if (length < 13) return filteredText.replace(/(\d{2})(\d{3})(\d{3})(\d{4})/, '$1.$2.$3/$4')
  else return filteredText.replace(/(\d{2})(\d{3})(\d{3})(\d{4})/, '$1.$2.$3/$4-')
}

export const formatCPF = (text: string) => {
  if (!text) return ''
  const filteredText = extractNumbers(text).slice(0, 11)
  const length = filteredText.length
  if (length < 4) return filteredText
  if (length < 7) return filteredText.replace(/(\d{3})/, '$1.')
  if (length < 10) return filteredText.replace(/(\d{3})(\d{3})/, '$1.$2.')
  return filteredText.replace(/(\d{3})(\d{3})(\d{3})/, '$1.$2.$3-')
}

export const formatFIPE = (text: string) => {
  if (!text) return ''
  const filteredText = extractNumbers(text).slice(0, 7)
  const length = filteredText.length
  if (length < 7) return filteredText
  return filteredText.replace(/(\d{6})/, '$1-')
}

export const formatCNPJ = (text: string) => {
  const filteredText = extractNumbers(text).slice(0, 14)
  const length = filteredText.length
  if (length < 2) return filteredText
  if (length < 6) return filteredText.replace(/(\d\d)/, '$1.')
  if (length < 9) return filteredText.replace(/(\d\d)(\d\d\d)/, '$1.$2.')
  if (length < 13) return filteredText.replace(/(\d\d)(\d\d\d)(\d\d\d)/, '$1.$2.$3/')
  if (length < 15) return filteredText.replace(/(\d\d)(\d\d\d)(\d\d\d)(\d\d\d\d)/, '$1.$2.$3/$4-')

  return text
}

export const formatCEP = (text: string) => {
  if (!text) return ''
  const filteredText = extractNumbers(text).slice(0, 8)
  const length = filteredText.length
  if (length < 6) return filteredText
  return filteredText.replace(/(\d{5})/, '$1-')
}

export const formatDateValue = (value: string | Date) => {
  if (!value) return null
  return typeof value === 'string' ? new Date(value) : value
}

export const formatDateText = (text?: string) => {
  if (!text) return ''
  const filteredString = extractNumbers(text).slice(0, 8)
  const length = filteredString.length
  if (length < 3) return filteredString
  if (length < 5) return filteredString.replace(/^(\d\d)/g, '$1/')
  if (length < 9) return filteredString.replace(/^(\d\d)(\d\d)/g, '$1/$2/')

  return text
}

const formatLeftSide = (text?: string) => {
  if (!text) return ''
  return text.replace(/\B(?=(\d{3})+(?!\d))/g, '.')
}

const formatRightSide = (text?: string) => {
  if (!text) return ''
  return text.padEnd(2, '0').slice(0, 2)
}

export const formatCurrency = (text?: string, partial = false, prefix = 'R$') => {
  if (!text) return ''
  const value = text.startsWith(prefix) ? extractCurrency(text) : text
  if (!value) return ''
  let [leftSide, rightSide] = value.split('.')
  if (!leftSide) {
    // usuário apertou '.' primeiro
    return `${prefix} 0,`
  }
  leftSide = formatLeftSide(leftSide)
  if (rightSide !== undefined) {
    rightSide = partial ? rightSide.slice(0, 2) : formatRightSide(rightSide)
    return `${prefix} ${leftSide},${rightSide}`
  } else {
    return `${prefix} ${leftSide}`
  }
}

export const formatNumber = (text?: string, partial = false) => {
  if (!text) return ''
  const value = extractDecimal(text)
  if (!value) return ''
  let [leftSide, rightSide] = value.split('.')
  if (!leftSide) {
    // usuário apertou '.' primeiro
    return `0,`
  }
  leftSide = formatLeftSide(leftSide)
  if (rightSide !== undefined) {
    rightSide = partial ? rightSide.slice(0, 2) : formatRightSide(rightSide)
    return `${leftSide},${rightSide}`
  } else {
    return `${leftSide}`
  }
}
