import React from 'react'

import { ProposalDataContext } from '@agiliza/api/domain'
import ImageUpload from '@agiliza/components/organisms/ImageUpload'
import RGTemplate from '@agiliza/components/templates/RGTemplate'
import { ActionType } from '@agiliza/utils/hooks/state'
import { extractNumbers, getBase64FromFile } from '@agiliza/utils/method'
import CPFCNPJField from '@agiliza/views/ProposalData/components/atoms/CPFCNPJField'
import { DatePicker, SelectField, SelectFieldProps, Switch } from '@curio/components'
import { TextField, TextFieldProps } from '@material-ui/core'
import { withStyles, WithStyles } from '@material-ui/core/styles'

import { CDSpousePersonalDataState } from '../state'
import styles from './styles'

type ExtendedProps = WithStyles<typeof styles>

interface Props extends ExtendedProps {
  state: CDSpousePersonalDataState
  actions: ActionType<CDSpousePersonalDataState>
  context: ProposalDataContext
}

const Form = (props: Props) => {
  const { classes, state, actions, context } = props

  const {
    cpf,
    name,
    birthday,
    marriageCertificate,
    rg,
    userCRAS,
    mothersName,
    fathersName,
    occupation,
    schooling,
    maritalStatus,
    participationPercentage,
    dependents,
  } = state

  const handleChangeDate = (type: keyof Pick<CDSpousePersonalDataState, 'birthday'>) => (date: Date | null) => {
    actions.update({ [type]: date })
  }

  const handleChangeText =
    (key: keyof CDSpousePersonalDataState): TextFieldProps['onChange'] =>
    (evt) => {
      actions.update({ [key]: evt.target.value })
    }

  const handleChangeProof: React.DetailedHTMLProps<React.InputHTMLAttributes<HTMLInputElement>, HTMLInputElement>['onChange'] = async (evt) => {
    const file = evt.target.files && evt.target.files[0]
    if (file) {
      const b64: string = await getBase64FromFile(file)
      actions.update({ marriageCertificate: b64 })
    }
  }

  const handleChangeRG = (rg: Partial<CDSpousePersonalDataState['rg']>) => {
    actions.update({ rg: rg as CDSpousePersonalDataState['rg'] })
  }

  const handleChangeSelect =
    (key: keyof CDSpousePersonalDataState): SelectFieldProps['onChange'] =>
    (value) => {
      actions.update({ [key]: value })
    }

  return (
    <form className={classes.form}>
      <CPFCNPJField value={cpf} onChange={handleChangeText('cpf')} onlyCPF />
      <ImageUpload label="Certidão de casamento" image={marriageCertificate} onChangeImage={handleChangeProof} />
      <TextField variant="outlined" label="Nome" value={name} onChange={handleChangeText('name')} />
      <RGTemplate rg={rg} onChange={handleChangeRG} />
      <DatePicker
        id="admission-date"
        label="Data de nascimento"
        value={birthday}
        onChange={handleChangeDate('birthday')}
        format="dd/MM/yyyy"
        className={classes.datePicker}
      />
      <Switch label="Usuário CRAS?" required value={userCRAS} onChange={(value) => actions.update({ userCRAS: value })} />
      <TextField variant="outlined" label="Nome mãe" value={mothersName} onChange={handleChangeText('mothersName')} />
      <TextField variant="outlined" label="Nome pai" value={fathersName} onChange={handleChangeText('fathersName')} />
      <TextField variant="outlined" label="Profissão" value={occupation} onChange={handleChangeText('occupation')} />
      <SelectField
        id="schooling-select-field"
        shrink={false}
        label="Escolaridade"
        value={schooling}
        items={context.schoolings.map((sch) => ({ label: sch.description, value: sch.id }))}
        onChange={handleChangeSelect('schooling')}
        variant="outlined"
      />
      <SelectField
        id="schooling-select-field"
        shrink={false}
        label="Estado Civil"
        value={maritalStatus}
        items={context.maritalStatus.map((sch) => ({ label: sch.description, value: sch.id }))}
        onChange={handleChangeSelect('maritalStatus')}
        variant="outlined"
      />
      <TextField
        variant="outlined"
        label="Participação (%)"
        value={extractNumbers(participationPercentage)}
        onChange={handleChangeText('participationPercentage')}
      />
      <TextField variant="outlined" label="Dependentes" value={extractNumbers(dependents)} onChange={handleChangeText('dependents')} />
    </form>
  )
}

export default withStyles(styles)(Form)
