import { Document, Project, ProjectStatus } from '@agiliza/api/domain'
import { Schedule } from '@agiliza/api/domain/schedule'
import {
  DocumentoApiModel,
  ProjetoAgenteApiModel,
  ProjetoSituacaoApiModel
} from '@microcredito/client'

import { DomainAdapter } from '../'
import { ScheduleMapper } from '../schedule'

export class ProjectMapper implements DomainAdapter<ProjetoAgenteApiModel, Project> {
  private statusAdapter: ProjectStatusMapper
  private scheduleAdapter: ScheduleMapper
  private documentAdapter: ProjectDocumentMapper
  constructor() {
    this.scheduleAdapter = new ScheduleMapper()
    this.statusAdapter = new ProjectStatusMapper()
    this.documentAdapter = new ProjectDocumentMapper()
  }
  public mapApiModelToDomain = (apimodel: ProjetoAgenteApiModel) => {
    const result: Project = {
      processNumber: apimodel.numeroProcesso,
      events: apimodel.eventos ?? [],
      id: apimodel.id,
      status: this.statusAdapter.mapApiModelToDomain(apimodel.situacao),
      schedules: apimodel.agendamentos?.map(this.scheduleAdapter.mapApiModelToDomain) || ([] as Schedule[]),
      documents: apimodel.documentos.map(this.documentAdapter.mapApiModelToDomain),
    }
    return result
  }
}

export class ProjectStatusMapper implements DomainAdapter<ProjetoSituacaoApiModel, ProjectStatus> {
  public mapApiModelToDomain = (apimodel: ProjetoSituacaoApiModel) => {
    const result: ProjectStatus = {
      id: apimodel.codigo,
      description: apimodel.descricao,
    }
    return result
  }
}

export class ProjectDocumentMapper implements DomainAdapter<DocumentoApiModel, Document> {
  public mapTipoDocumentoApiToDomain = (tipo: DocumentoApiModel['tipoDocumento']): Document['type'] => ({
    id: tipo.codigo,
    description: tipo.descricao,
  })
  public mapApiModelToDomain = (apimodel: DocumentoApiModel): Document => ({
    id: apimodel.id || '',
    url: apimodel.imagem,
    type: this.mapTipoDocumentoApiToDomain(apimodel.tipoDocumento),
    extraInformations: apimodel.informacoesComplementares || '',
  })
}
