import React from 'react'

import { Workplace } from '@agiliza/api/domain'
import ImageUpload from '@agiliza/components/organisms/ImageUpload'
import PhoneField from '@agiliza/components/organisms/PhoneField'
import { formatCurrency } from '@agiliza/utils/formatters'
import { ActionType } from '@agiliza/utils/hooks/state'
import { maskLandline } from '@agiliza/utils/masks'
import { getBase64FromFile } from '@agiliza/utils/method'
import CPFCNPJField from '@agiliza/views/ProposalData/components/atoms/CPFCNPJField'
import { DatePicker } from '@curio/components'
import { TextField, TextFieldProps, Typography } from '@material-ui/core'
import { withStyles, WithStyles } from '@material-ui/core/styles'

import { CDSpousePersonalDataState } from '../../SpouseData/state'
import styles from './styles'

type ExtendedProps = WithStyles<typeof styles>

interface Props extends ExtendedProps {
  state: CDSpousePersonalDataState
  actions: ActionType<CDSpousePersonalDataState>
}

const Form = (props: Props) => {
  const { classes, state, actions } = props

  const { mobile, landline, primaryEmail, secundaryEmail, workplace } = state

  const handleChangeText =
    (key: keyof CDSpousePersonalDataState): TextFieldProps['onChange'] =>
    (evt) => {
      actions.update({ [key]: evt.target.value })
    }

  const handleChangePhoto = async (evt) => {
    const file = evt.target.files && evt.target.files[0]
    if (file) {
      const b64: string = await getBase64FromFile(file)
      actions.update({ workplace: { proof: b64 } })
    }
  }

  const handleChangeWorkplace =
    (key: keyof Workplace): TextFieldProps['onChange'] =>
    (evt) => {
      actions.update({ workplace: { [key]: evt.target.value } })
    }

  const handleChangeDate = (key: keyof Pick<Workplace, 'admissionDate'>) => (date: Date | null) => {
    actions.update({ workplace: { [key]: date } })
  }

  return (
    <form className={classes.form}>
      <PhoneField label="Telefone celular" value={mobile} onChange={handleChangeText('mobile')} />
      <TextField variant="outlined" label="Telefone fixo" value={maskLandline(landline)} onChange={handleChangeText('landline')} />
      <TextField variant="outlined" label="Email primário" value={primaryEmail} onChange={handleChangeText('primaryEmail')} />
      <TextField variant="outlined" label="Email secundário" value={secundaryEmail} onChange={handleChangeText('secundaryEmail')} />
      <Typography className={classes.workplaceTitle} variant="h6">
        Local de trabalho
      </Typography>
      <TextField variant="outlined" label="Razão social" value={workplace?.workplaceName || ''} onChange={handleChangeWorkplace('workplaceName')} />
      <ImageUpload label="Comprovante" image={workplace?.proof || ''} onChangeImage={handleChangePhoto} />
      <CPFCNPJField value={workplace?.cnpj || ''} onChange={handleChangeWorkplace('cnpj')} onlyCPNJ />
      <DatePicker
        id="admission-date"
        label="Data de admissão"
        value={workplace?.admissionDate}
        onChange={handleChangeDate('admissionDate')}
        format="dd/MM/yyyy"
        className={classes.datePicker}
      />
      <PhoneField value={workplace?.phone} onChange={handleChangeWorkplace('phone')} />
      <TextField
        variant="outlined"
        label="Renda líquida"
        value={formatCurrency(workplace?.netIncome || '')}
        onChange={handleChangeWorkplace('netIncome')}
      />
      <TextField variant="outlined" label="Cargo ou função" value={workplace?.role} onChange={handleChangeWorkplace('role')} />
    </form>
  )
}

export default withStyles(styles)(Form)
