import React from 'react'

import { ProposalDataContext } from '@agiliza/api/domain'
import PhoneField from '@agiliza/components/organisms/PhoneField'
import RGTemplate from '@agiliza/components/templates/RGTemplate'
import { ActionType } from '@agiliza/utils/hooks/state'
import { maskLandline } from '@agiliza/utils/masks'
import { DatePicker, SelectField, SelectFieldProps, Switch } from '@curio/components'
import { TextField, TextFieldProps } from '@material-ui/core'
import { withStyles, WithStyles } from '@material-ui/core/styles'

import { CDPersonalDataState } from '../state'
import styles from './styles'

type ExtendedProps = WithStyles<typeof styles>

interface Props extends ExtendedProps {
  state: CDPersonalDataState
  actions: ActionType<CDPersonalDataState>
  context: ProposalDataContext
}

const Form = (props: Props) => {
  const { classes, state, actions, context } = props

  const {
    birthday,
    primaryEmail,
    rg,
    userCRAS,
    mothersName,
    fathersName,
    occupation,
    schooling,
    maritalStatus,
    dependents,
    secundaryEmail,
    mobile,
    landline,
  } = state

  const handleChangeDate = (type: keyof Pick<CDPersonalDataState, 'birthday'>) => (date: Date | null) => {
    actions.update({ [type]: date })
  }

  const handleChangeText =
    (key: keyof CDPersonalDataState): TextFieldProps['onChange'] =>
    (evt) => {
      actions.update({ [key]: evt.target.value })
    }

  const handleChangeRG = (rg: Partial<CDPersonalDataState['rg']>) => {
    actions.update({ rg: rg as CDPersonalDataState['rg'] })
  }

  const handleChangeSelect =
    (key: keyof CDPersonalDataState): SelectFieldProps['onChange'] =>
    (value) => {
      actions.update({ [key]: value })
    }

  return (
    <form className={classes.form}>
      <DatePicker
        id="birthday"
        label="Data de nascimento"
        value={birthday}
        onChange={handleChangeDate('birthday')}
        format="dd/MM/yyyy"
        className={classes.datePicker}
      />
      <RGTemplate rg={rg} onChange={handleChangeRG} />
      <Switch label="Usuário CRAS?" required value={userCRAS} onChange={(value) => actions.update({ userCRAS: value })} />
      <TextField variant="outlined" label="Nome da mãe" value={mothersName} onChange={handleChangeText('mothersName')} />
      <TextField variant="outlined" label="Nome do pai" value={fathersName} onChange={handleChangeText('fathersName')} />
      <TextField variant="outlined" label="Profissão" value={occupation} onChange={handleChangeText('occupation')} />
      <SelectField
        id="schooling-select-field"
        shrink={false}
        label="Escolaridade"
        value={schooling}
        items={context.schoolings.map((sch) => ({ label: sch.description, value: sch.id }))}
        onChange={handleChangeSelect('schooling')}
        variant="outlined"
      />
      <SelectField
        id="schooling-select-field"
        shrink={false}
        label="Estado Civil"
        value={maritalStatus}
        items={context.maritalStatus.map((sch) => ({ label: sch.description, value: sch.id }))}
        onChange={handleChangeSelect('maritalStatus')}
        variant="outlined"
      />
      <TextField variant="outlined" label="Dependentes" type="number" value={dependents} onChange={handleChangeText('dependents')} />
      <TextField variant="outlined" label="Email primário" value={primaryEmail} onChange={handleChangeText('primaryEmail')} />
      <TextField variant="outlined" label="Email secondário" value={secundaryEmail} onChange={handleChangeText('secundaryEmail')} />
      <PhoneField label="Telefone celular" value={mobile} onChange={handleChangeText('mobile')} />
      <TextField
        variant="outlined"
        label="Telefone fixo"
        value={maskLandline(landline)}
        onChange={handleChangeText('landline')}
        inputProps={{ maxLength: 14 }}
      />
    </form>
  )
}

export default withStyles(styles)(Form)
