import { ErrorTypes } from '@agiliza/constants/error'

interface Error {
  type: ErrorTypes
  omit?: boolean
}

// Usuário não está autorizado (Login)
export interface UnauthorizedError extends Error {
  type: ErrorTypes.UNAUTHORIZED
  message?: string
}

// Token de usuário expirou
export interface ExpiredSessionError extends Error {
  type: ErrorTypes.EXPIRED_SESSION
  message: string
}

// Validação de campos (422, resposta padrão para esse status)
export interface InvalidFields {
  [field: string]: string
}

export interface FormValidationError extends Error {
  type: ErrorTypes.FORM_VALIDATION
  invalidFields: InvalidFields
  detail: string
}

// Campo ausente do requerido na API (não deve aparecer normalmente)
export interface MissingInputError extends Error {
  type: ErrorTypes.MISSING_INPUT
  missingFields: string[]
  detail: string
}

// Erros de servidor (500) e quais outros erros não mapeados
export interface InternalServerError extends Error {
  type: ErrorTypes.INTERNAL_SERVER
  message: string
  stack: string
}

// Código de Primeiro Acesso/Esqueci a Senha é inválido
export interface InvalidVerificationCodeError extends Error {
  type: ErrorTypes.INVALID_EMAIL_CODE
  message: string
}

// Usuário já existe disponível no sistema (Primeiro Acesso)
export interface AlreadyEnabledError extends Error {
  type: ErrorTypes.ALREADY_ENABLED
  message?: string
}

// Usuário não encontrado (404)
export interface UserNotFoundError extends Error {
  type: ErrorTypes.USER_NOT_FOUND
  message?: string
}

// Usuário não pode agendar uma nova visita
export interface ScheduleForbiddenError extends Error {
  type: ErrorTypes.SCHEDULE_FORBIDDEN
  message?: string
}

export type ApiError =
  | UnauthorizedError
  | ExpiredSessionError
  | FormValidationError
  | InternalServerError
  | MissingInputError
  | InvalidVerificationCodeError
  | AlreadyEnabledError
  | UserNotFoundError
  | ScheduleForbiddenError
