import {
  BDIdentification,
  BDProductService,
  Document,
  StockSalesCosts,
  UpdateBDAddress
} from '@agiliza/api/domain'
import {
  DadosNegocioEnderecoApiModel,
  DadosNegocioEstoqueVendasCustosApiModel,
  DadosNegocioIdentificacaoApiModel,
  DadosNegocioProdutoServicoApiModel,
  DocumentoApiModel
} from '@microcredito/client'

import { DomainAdapter } from '../shared'
import { AddressMapper } from './customerData'

export class ProjectDocumentMapper implements DomainAdapter<DocumentoApiModel, Document> {
  public mapTipoDocumentoApiToDomain = (tipo: DocumentoApiModel['tipoDocumento']): Document['type'] => ({
    id: tipo.codigo,
    description: tipo.descricao,
  })
  public mapApiModelToDomain = (apimodel: DocumentoApiModel): Document => ({
    id: apimodel.id,
    url: apimodel.imagem,
    type: this.mapTipoDocumentoApiToDomain(apimodel.tipoDocumento),
    extraInformations: apimodel.informacoesComplementares || '',
  })
}

export class BDIdentificationMapper implements DomainAdapter<DadosNegocioIdentificacaoApiModel, BDIdentification> {
  private documentAdapter: ProjectDocumentMapper

  constructor() {
    this.documentAdapter = new ProjectDocumentMapper()
  }
  public mapApiModelToDomain = (dnIdent: DadosNegocioIdentificacaoApiModel): BDIdentification => ({
    activitySector: dnIdent.setorAtividade || '',
    cityRegistration: dnIdent.inscricaoMunicipal || '',
    cnpj: dnIdent.cnpj || '',
    codeCNAE: dnIdent.codigoCNAE || '',
    commercialRegistration: dnIdent.inscricaoComercial || '',
    companyName: dnIdent.razaoSocial || '',
    fantasyName: dnIdent.nomeFantasia || '',
    local: dnIdent.local || '',
    openingDate: dnIdent.dataAbertura || null,
    settledOP: dnIdent.opLiquidada || '',
    stateRegistration: dnIdent.inscricaoEstadual || '',
    document: dnIdent.documento ? this.documentAdapter.mapApiModelToDomain(dnIdent.documento) : undefined,
  })
}

export class BDProductsServicesMapper implements DomainAdapter<DadosNegocioProdutoServicoApiModel, BDProductService> {
  public mapPriceAmountToDomain = (pA: DadosNegocioProdutoServicoApiModel['custosVariaveis']) => ({
    price: Number(pA?.precoUnitario).toFixed(2),
    amount: pA?.quantidade?.toString() || '',
  })
  public mapApiModelToDomain = (pS: DadosNegocioProdutoServicoApiModel): BDProductService => ({
    id: pS.id || '',
    description: pS.descricao || '',
    unit: pS.unidade || '',
    monthlySales: pS.vendasMensais ? this.mapPriceAmountToDomain(pS.vendasMensais) : { amount: '', price: '' },
    stock: pS.estoque ? this.mapPriceAmountToDomain(pS.estoque) : { amount: '', price: '' },
    variableCosts: pS.custosVariaveis ? this.mapPriceAmountToDomain(pS.custosVariaveis) : { amount: '', price: '' },
  })
}

export class BDUpdateAddressMapper implements DomainAdapter<DadosNegocioEnderecoApiModel, UpdateBDAddress['Output']> {
  private bdAddressMapper: AddressMapper
  constructor() {
    this.bdAddressMapper = new AddressMapper()
  }
  public mapApiModelToDomain = (bdAdr: DadosNegocioEnderecoApiModel): UpdateBDAddress['Output'] => ({
    proof: bdAdr.comprovante || '',
    referencePoint: bdAdr.pontoReferencia,
    ...this.bdAddressMapper.mapApiModelToDomain(
      bdAdr.endereco || { bairro: '', cep: '', estado: '', logradouro: '', municipio: '', numero: '', complemento: '' }
    ),
  })
}

export class BDUpdateStockSalesCostsMapper implements DomainAdapter<DadosNegocioEstoqueVendasCustosApiModel, StockSalesCosts> {
  private documentMapper: ProjectDocumentMapper
  constructor() {
    this.documentMapper = new ProjectDocumentMapper()
  }

  public mapApiModelToDomain = (sSC: DadosNegocioEstoqueVendasCustosApiModel): StockSalesCosts => ({
    document: sSC.documento && this.documentMapper.mapApiModelToDomain(sSC.documento),
    monthRating: sSC.classificacaoMes || '',
    monthYearReference: sSC.mesAnoReferencia || '',
    salesCommission: sSC.comissaoVendas?.toString() || '',
  })
}
